<?php
/**
 * Pointify - Listing Directory WordPress Theme functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Pointify_-_Listing_Directory_WordPress_Theme
 */

if ( ! defined( '_S_VERSION' ) ) {
	// Replace the version number of the theme on each release.
	define( '_S_VERSION', '1.0.0' );
}

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function pointify_setup() {
	/*
		* Make theme available for translation.
		* Translations can be filed in the /languages/ directory.
		* If you're building a theme based on Pointify - Listing Directory WordPress Theme, use a find and replace
		* to change 'pointify' to the name of your theme in all the template files.
		*/
	load_theme_textdomain( 'pointify', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
		* Let WordPress manage the document title.
		* By adding theme support, we declare that this theme does not use a
		* hard-coded <title> tag in the document head, and expect WordPress to
		* provide it for us.
		*/
	add_theme_support( 'title-tag' );

	/**
	 * Add Theme Support functions
	 */

	add_theme_support( "wp-block-styles" );
	add_theme_support( "responsive-embeds" );
	add_theme_support( "align-wide" );
	add_theme_support( "register_block_style" );
	add_theme_support( "register_block_pattern" );
	add_editor_style('editor-style.css');

	/*
		* Enable support for Post Thumbnails on posts and pages.
		*
		* @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		*/
	add_theme_support( 'post-thumbnails' );

	// This theme uses wp_nav_menu() in one location.
	register_nav_menus(
		array(
			'menu-1' => esc_html__( 'Primary', 'pointify' ),
			'tertiary'  => __( 'Tertiary Menu', 'pointify' ),
			'user-menu' => esc_html__( 'User Menu', 'pointify' ),
		)
	);

	/*
		* Switch default core markup for search form, comment form, and comments
		* to output valid HTML5.
		*/
	add_theme_support(
		'html5',
		array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
			'style',
			'script',
		)
	);

	// Set up the WordPress core custom background feature.
	add_theme_support(
		'custom-background',
		apply_filters(
			'pointify_custom_background_args',
			array(
				'default-color' => 'ffffff',
				'default-image' => '',
			)
		)
	);

	// Add theme support for selective refresh for widgets.
	add_theme_support( 'customize-selective-refresh-widgets' );
	add_theme_support( 'job-manager-templates' );

	/**
	 * Add support for core custom logo.
	 *
	 * @link https://codex.wordpress.org/Theme_Logo
	 */
	add_theme_support(
		'custom-logo',
		array(
			'height'      => 250,
			'width'       => 250,
			'flex-width'  => true,
			'flex-height' => true,
		)
	);
}
add_action( 'after_setup_theme', 'pointify_setup' );


// Define custom menu fallback
function pointify_custom_menu_fallback() {
	echo 'Create Main Menu by Click on <a href="' . esc_url( admin_url( 'nav-menus.php' ) ) . '">Create Menu</a>';
}

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function pointify_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'pointify_content_width', 640 );
}
add_action( 'after_setup_theme', 'pointify_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function pointify_widgets_init() {
	register_sidebar(
		array(
			'name'          => esc_html__( 'Sidebar', 'pointify' ),
			'id'            => 'sidebar-1',
			'description'   => esc_html__( 'Add widgets here.', 'pointify' ),
			'before_widget' => '<section id="%1$s" class="widget %2$s">',
			'after_widget'  => '</section>',
			'before_title'  => '<h2 class="widget-title">',
			'after_title'   => '</h2>',
		)
	);
}
add_action( 'widgets_init', 'pointify_widgets_init' );

/**
 * Enqueue scripts and styles.
 */
function pointify_scripts() {

	// Bootstrap CSS
    wp_enqueue_style('bootstrap-css', get_template_directory_uri() . '/assets/bootstrap/css/bootstrap.min.css', array(), '5.3.2', 'all');

	// Font Awesome CSS
    wp_enqueue_style('font-awesome-min', get_template_directory_uri() . '/assets/font-awesome/css/font-awesome.min.css', array(), '4.7.0', 'all');

	//Owl Carousel

	wp_enqueue_style('owl-carousel-css', get_template_directory_uri() . '/assets/owl-carousel/css/owl.carousel.min.css', array(), '2.3.4', 'all');

	//Swiper
	wp_enqueue_style('swiper-bundle-min-css', get_template_directory_uri() . '/assets/swiper/css/swiper-bundle.min.css', array(), '9.1.0', 'all');

	// Theme Style
    wp_enqueue_style('default-style', get_template_directory_uri() . '/assets/theme-style/default-style.css', array(), 'all');
    wp_enqueue_style('default-responsive-style', get_template_directory_uri() . '/assets/theme-style/default-responsive-style.css', array(), 'all');

	$style_demo = get_theme_mod( 'pointify_demo_selection', '' );

	if($style_demo === 'job_board'){
		wp_enqueue_style('job-board-style', get_template_directory_uri() . '/assets/theme-style/job-board-style.css', array(), 'all');
		wp_enqueue_style('job-board-responsive-style', get_template_directory_uri() . '/assets/theme-style/job-board-responsive-style.css', array(), 'all');
	}

	if($style_demo === 'lawyer'){
		wp_enqueue_style('lawyer-style', get_template_directory_uri() . '/assets/theme-style/lawyer-style.css', array(), 'all');
		wp_enqueue_style('lawyer-responsive-style', get_template_directory_uri() . '/assets/theme-style/lawyer-responsive-style.css', array(), 'all');
	}

	if (is_rtl()) {
		wp_enqueue_style('pointify-style-rtl', get_template_directory_uri() . '/style-rtl.css', array(), 'all');
	}

	// Bootstrap JS
	wp_enqueue_script('bootstrap-js', get_template_directory_uri() . '/assets/bootstrap/js/bootstrap.min.js', array('jquery'), '5.3.2', true);

	wp_enqueue_script('owl-carousel-js', get_template_directory_uri() . '/assets/owl-carousel/js/owl.carousel.min.js', array('jquery'),'2.3.4', true);

	wp_enqueue_script('swiper-bundle-min-js', get_template_directory_uri() . '/assets/swiper/js/swiper-bundle.min.js', array('jquery'),'9.1.0', true);

	wp_enqueue_script('jquery-magnific-popup-min-js', get_template_directory_uri() . '/assets/magnific-popup/jquery.magnific-popup.min.js', array('jquery'),'1.1.0', true);

	wp_enqueue_script('jquery-masonry-pkgd-min-js', get_template_directory_uri() . '/assets/masonry/masonry.pkgd.min.js', array('jquery'),'4.2.2', false);

	//Theme JS
    wp_enqueue_script('pointify-theme', get_template_directory_uri() . '/assets/theme-js/pointify-theme.js', array('jquery'), true);
	

	wp_localize_script('pointify-theme', 'custom_like_vars', array(
        'post_id' => get_the_ID(),
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('custom-like-nonce'),
    ));

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
	
}
add_action( 'wp_enqueue_scripts', 'pointify_scripts' );

/**
 * Admin scripts
 */
function pointify_admin_scripts() {

	// Upload script, require `wp_enqueue_media()`.
	wp_register_script( 'pointify-admin-widget-media', get_template_directory_uri() . '/js/widgets/widget-media.js', array( 'jquery' ) );
}
add_action( 'admin_enqueue_scripts', 'pointify_admin_scripts' );

/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/template-functions.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Theme Plugin
 */
require get_template_directory() . '/inc/theme-plugins.php';

/**
 * Load Jetpack compatibility file.
 */
if ( defined( 'JETPACK__VERSION' ) ) {
	require get_template_directory() . '/inc/jetpack.php';
}

/**
 * Load WooCommerce compatibility file.
 */
if ( class_exists( 'WooCommerce' ) ) {
	require get_template_directory() . '/inc/woocommerce.php';
}

/**
 * Theme Register File
 */

if ( class_exists( 'WP_Job_Manager' ) ) {
	require get_template_directory() . '/inc/includes/wpjm-geolocation.php';
}
require get_template_directory() . '/inc/tgmpa/class-tgm-plugin-activation.php';
require get_template_directory() . '/inc/tgmpa/extras.php';
require get_template_directory() . '/inc/demos/one-click-demo-import.php';
require get_template_directory() . '/inc/widget/register-sidebar-widget.php';
require get_template_directory() . '/inc/widget/latest-post-widget.php';
require get_template_directory() . '/inc/customizer/sections/customizer-settings.php';
require get_template_directory() . '/inc/theme-svg.php';

if ( class_exists( 'Extensify' ) ) {
	require get_template_directory() . '/inc/customizer/customizer-color.php'; 
	require get_template_directory() . '/inc/widget/job-single-location-widget.php';
	require get_template_directory() . '/inc/widget/job-single-map-widget.php';
	if ( get_option( 'job_manager_enable_categories' ) == 1 ) {
		require get_template_directory() . '/inc/widget/page-widgets/widget-tabbed-listings.php';
	}
	require get_template_directory() . '/inc/widget/page-widgets/widget-pricing-table.php';
}

require get_template_directory() . '/inc/widget/user-carousel-widget.php';
require get_template_directory() . '/inc/widget/instagram-widget.php';
require get_template_directory() . '/inc/comments-form-placeholder.php';
/* Page Widgets files */
require get_template_directory() . '/inc/widget/page-widgets/widget-search-hero.php';
require get_template_directory() . '/inc/widget/page-widgets/widget-popular-listing.php';
require get_template_directory() . '/inc/widget/page-widgets/widget-latest-listing.php';
require get_template_directory() . '/inc/widget/page-widgets/widget-how-it-work.php';
require get_template_directory() . '/inc/widget/page-widgets/widget-cta-section.php';
require get_template_directory() . '/inc/widget/page-widgets/widget-blog-posts.php';
require get_template_directory() . '/inc/widget/page-widgets/widget-listing-review.php';
require get_template_directory() . '/inc/widget/page-widgets/widget-image-content.php';

if ( get_option( 'job_manager_enable_categories' ) == 1 ) {
	require get_template_directory() . '/inc/widget/page-widgets/widget-home-taxonomy-image-grid.php';
}

$style_demo = get_theme_mod( 'pointify_demo_selection', '' );

if($style_demo === 'job_board'){
	require get_template_directory() . '/inc/widget/page-widgets/widget-how-it-work-page.php';
}

/**
 * Pointify Demo Class
 */

function pointify_body_class( $classes ) {
    $demo = get_theme_mod( 'pointify_demo_selection', 'business' );

    if ( $demo ) {
        $classes[] = $demo;
    }

    return $classes;
}
add_filter( 'body_class', 'pointify_body_class' );

/**
 * Pointify Search
 */

function pointify_search_only_posts( $query ) {
    // Check if it is a search query and if it's the main query
    if ( $query->is_search && $query->is_main_query() && !is_admin() ) {
        $query->set( 'post_type', 'post' );
    }
}
add_action( 'pre_get_posts', 'pointify_search_only_posts' );


// Pointify Related Job

function pointify_get_related_jobs($post_id, $limit = 5) {
    // Get the job categories for the current job
    $terms = wp_get_post_terms($post_id, 'job_listing_category', array('fields' => 'ids'));
    
    if (empty($terms)) {
        return [];
    }

    // Query arguments to fetch related jobs
    $args = array(
        'post_type' => 'job_listing',
        'posts_per_page' => $limit,
        'post__not_in' => array($post_id),
        'tax_query' => array(
            array(
                'taxonomy' => 'job_listing_category',
                'field' => 'term_id',
                'terms' => $terms,
            ),
        ),
    );

    // Fetch the related jobs
    $related_jobs = new WP_Query($args);

    return $related_jobs;
}
