<?php
/**
 * Theme functions and definitions.
 *
 * @link https://codex.wordpress.org/Functions_File_Explained
 *
 * @package mizan_real_estate
 */

if ( ! function_exists( 'mizan_real_estate_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function mizan_real_estate_setup() {
		/*
		 * Make theme available for translation.
		 */ 
		load_theme_textdomain( 'mizan-real-estate', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		// Let WordPress manage the document title.
		add_theme_support( 'title-tag' );

		// Enable support for Post Thumbnails on posts and pages.
		add_theme_support( 'post-thumbnails' );
		add_image_size( 'mizan-real-estate-thumb', 400, 300 );

		// Register nav menu locations.
		register_nav_menus( array(
			'primary-menu'  => esc_html__( 'Primary Menu', 'mizan-real-estate' ),
		) );

		/*
		* This theme styles the visual editor to resemble the theme style,
		* specifically font, colors, icons, and column width.
		*/
		$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
		add_editor_style( array( '/css/editor-style' . $min . '.css', mizan_real_estate_fonts_url() ) );

		/*
		 * Switch default core markup to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		// Set up the WordPress core custom background feature.
		add_theme_support( 'custom-background', apply_filters( 'mizan_real_estate_custom_background_args', array(
			'default-color' => 'f7fcfe',
		) ) );

		add_theme_support( 'align-wide' );

		// Enable support for selective refresh of widgets in Customizer.
		add_theme_support( 'customize-selective-refresh-widgets' );

		// Enable support for custom logo.
		add_theme_support( 'custom-logo', array(
			'height'      => 240,
			'width'       => 240,
			'flex-height' => true,
		) );

		// Load default block styles.
		add_theme_support( 'wp-block-styles' );

		// Add support for responsive embeds.
		add_theme_support( 'responsive-embeds' );

		// Enable support for footer widgets.
		add_theme_support( 'footer-widgets', 4 );

		// woocommerce
		add_theme_support( 'woocommerce' );

		// Load Supports.
		require_once get_template_directory() . '/inc/support.php';

		// Add custom editor font sizes.
		add_theme_support(
			'editor-font-sizes',
			array(
				array(
					'name'      => __( 'Small', 'mizan-real-estate' ),
					'shortName' => __( 'S', 'mizan-real-estate' ),
					'size'      => 13,
					'slug'      => 'small',
				),
				array(
					'name'      => __( 'Normal', 'mizan-real-estate' ),
					'shortName' => __( 'M', 'mizan-real-estate' ),
					'size'      => 14,
					'slug'      => 'normal',
				),
				array(
					'name'      => __( 'Large', 'mizan-real-estate' ),
					'shortName' => __( 'L', 'mizan-real-estate' ),
					'size'      => 30,
					'slug'      => 'large',
				),
				array(
					'name'      => __( 'Huge', 'mizan-real-estate' ),
					'shortName' => __( 'XL', 'mizan-real-estate' ),
					'size'      => 36,
					'slug'      => 'huge',
				),
			)
		);

		// Editor color palette.
		add_theme_support(
			'editor-color-palette',
			array(
				array(
					'name'  => __( 'Black', 'mizan-real-estate' ),
					'slug'  => 'black',
					'color' => '#121212',
				),
				array(
					'name'  => __( 'White', 'mizan-real-estate' ),
					'slug'  => 'white',
					'color' => '#ffffff',
				),
				array(
					'name'  => __( 'Gray', 'mizan-real-estate' ),
					'slug'  => 'gray',
					'color' => '#727272',
				),
				array(
					'name'  => __( 'Blue', 'mizan-real-estate' ),
					'slug'  => 'blue',
					'color' => '#007BFF',
				),
				array(
					'name'  => __( 'Navy Blue', 'mizan-real-estate' ),
					'slug'  => 'navy-blue',
					'color' => '#007BFF',
				),
				array(
					'name'  => __( 'Light Blue', 'mizan-real-estate' ),
					'slug'  => 'light-blue',
					'color' => '#f7fcfe',
				),
				array(
					'name'  => __( 'Orange', 'mizan-real-estate' ),
					'slug'  => 'orange',
					'color' => '#121212',
				),
				array(
					'name'  => __( 'Green', 'mizan-real-estate' ),
					'slug'  => 'green',
					'color' => '#77a464',
				),
				array(
					'name'  => __( 'Red', 'mizan-real-estate' ),
					'slug'  => 'red',
					'color' => '#e4572e',
				),
				array(
					'name'  => __( 'Yellow', 'mizan-real-estate' ),
					'slug'  => 'yellow',
					'color' => '#f4a024',
				),
			)
		);
	}
endif;

add_action( 'after_setup_theme', 'mizan_real_estate_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function mizan_real_estate_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'mizan_real_estate_content_width', 771 );
}
add_action( 'after_setup_theme', 'mizan_real_estate_content_width', 0 );

/**
 * Register widget area.
 */
function mizan_real_estate_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Primary Sidebar', 'mizan-real-estate' ),
		'id'            => 'sidebar-1',
		'description'   => esc_html__( 'Add widgets here to appear in your Primary Sidebar.', 'mizan-real-estate' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Secondary Sidebar', 'mizan-real-estate' ),
		'id'            => 'sidebar-2',
		'description'   => esc_html__( 'Add widgets here to appear in your Secondary Sidebar.', 'mizan-real-estate' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Secondary Sidebar 1', 'mizan-real-estate' ),
		'id'            => 'sidebar-3',
		'description'   => esc_html__( 'Add widgets here to appear in your Secondary Sidebar 1.', 'mizan-real-estate' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
}
add_action( 'widgets_init', 'mizan_real_estate_widgets_init' );

/**
 * Register custom fonts.
 */
function mizan_real_estate_fonts_url() {
	$font_family   = array(
		'Hind+Madurai:wght@300;400;500;600;700',
		'Josefin+Sans:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;1,100;1,200;1,300;1,400;1,500;1,600;1,700',
		'Poppins:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900',
		'Bad Script',
		'Bebas Neue',
		'Fjalla One',
		'PT Sans:ital,wght@0,400;0,700;1,400;1,700',
		'PT Serif:ital,wght@0,400;0,700;1,400;1,700',
		'Roboto:ital,wght@0,100;0,300;0,400;0,500;0,700;0,900;1,100;1,300;1,400;1,500;1,700;1,900',
		'Roboto Condensed:ital,wght@0,300;0,400;0,700;1,300;1,400;1,700',
		'Alex Brush',
		'Overpass:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900',
		'Montserrat:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900',
		'Playball',
		'Alegreya:ital,wght@0,400;0,500;0,600;0,700;0,800;0,900;1,400;1,500;1,600;1,700;1,800;1,900',
		'Julius Sans One',
		'Arsenal:ital,wght@0,400;0,700;1,400;1,700',
		'Slabo 13px',
		'Lato:ital,wght@0,100;0,300;0,400;0,700;0,900;1,100;1,300;1,400;1,700;1,900',
		'Overpass Mono:wght@300;400;500;600;700',
		'Source Sans Pro:ital,wght@0,200;0,300;0,400;0,600;0,700;0,900;1,200;1,300;1,400;1,600;1,700;1,900',
		'Raleway:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900',
		'Merriweather:ital,wght@0,300;0,400;0,700;0,900;1,300;1,400;1,700;1,900',
		'Rubik:ital,wght@0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,300;1,400;1,500;1,600;1,700;1,800;1,900',
		'Lora:ital,wght@0,400;0,500;0,600;0,700;1,400;1,500;1,600;1,700',
		'Ubuntu:ital,wght@0,300;0,400;0,500;0,700;1,300;1,400;1,500;1,700',
		'Cabin:ital,wght@0,400;0,500;0,600;0,700;1,400;1,500;1,600;1,700',
		'Arimo:ital,wght@0,400;0,500;0,600;0,700;1,400;1,500;1,600;1,700',
		'Playfair Display:ital,wght@0,400;0,500;0,600;0,700;0,800;0,900;1,400;1,500;1,600;1,700;1,800;1,900',
		'Quicksand:wght@300..700',
		'Padauk:wght@400;700',
		'Mulish:ital,wght@0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;0,1000;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900;1,1000',
		'Inconsolata:wght@200;300;400;500;600;700;800;900&family=Mulish:ital,wght@0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;0,1000;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900;1,1000',
		'Bitter:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900&family=Mulish:ital,wght@0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;0,1000;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900;1,1000',
		'Pacifico',
		'Indie Flower',
		'VT323',
		'Dosis:wght@200;300;400;500;600;700;800',
		'Frank Ruhl Libre:wght@300;400;500;700;900',
		'Fjalla One',
		'Figtree:ital,wght@0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,300;1,400;1,500;1,600;1,700;1,800;1,900',
		'Oxygen:wght@300;400;700',
		'Arvo:ital,wght@0,400;0,700;1,400;1,700',
		'Noto Serif:ital,wght@0,400;0,700;1,400;1,700',
		'Lobster',
		'Crimson Text:ital,wght@0,400;0,600;0,700;1,400;1,600;1,700',
		'Yanone Kaffeesatz:wght@200;300;400;500;600;700',
		'Anton',
		'Libre Baskerville:ital,wght@0,400;0,700;1,400',
		'Bree Serif',
		'Gloria Hallelujah',
		'Abril Fatface',
		'Varela Round',
		'Vampiro One',
		'Shadows Into Light',
		'Cuprum:ital,wght@0,400;0,500;0,600;0,700;1,400;1,500;1,600;1,700',
		'Rokkitt:wght@100;200;300;400;500;600;700;800;900',
		'Vollkorn:ital,wght@0,400;0,500;0,600;0,700;0,800;0,900;1,400;1,500;1,600;1,700;1,800;1,900',
		'Francois One',
		'Orbitron:wght@400;500;600;700;800;900',
		'Patua One',
		'Acme',
		'Satisfy',
		'Josefin Slab:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;1,100;1,200;1,300;1,400;1,500;1,600;1,700',
		'Quattrocento Sans:ital,wght@0,400;0,700;1,400;1,700',
		'Architects Daughter',
		'Russo One',
		'Monda:wght@400;700',
		'Righteous',
		'Lobster Two:ital,wght@0,400;0,700;1,400;1,700',
		'Hammersmith One',
		'Courgette',
		'Permanent Marke',
		'Cherry Swash:wght@400;700',
		'Cormorant Garamond:ital,wght@0,300;0,400;0,500;0,600;0,700;1,300;1,400;1,500;1,600;1,700',
		'Poiret One',
		'BenchNine:wght@300;400;700',
		'Economica:ital,wght@0,400;0,700;1,400;1,700',
		'Handlee',
		'Cardo:ital,wght@0,400;0,700;1,400',
		'Alfa Slab One',
		'Averia Serif Libre:ital,wght@0,300;0,400;0,700;1,300;1,400;1,700',
		'Cookie',
		'Chewy',
		'Great Vibes',
		'Coming Soon',
		'Philosopher:ital,wght@0,400;0,700;1,400;1,700',
		'Days One',
		'Kanit:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900',
		'Shrikhand',
		'Tangerine:wght@400;700',
		'IM Fell English SC',
		'Boogaloo',
		'Bangers',
		'Fredoka One',
		'Volkhov:ital,wght@0,400;0,700;1,400;1,700',
		'Shadows Into Light Two',
		'Marck Script',
		'Sacramento',
		'Unica One',
		'Dancing Script:wght@400;500;600;700',
		'Exo 2:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900',
		'Archivo:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900',
		'Jost:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900',
		'DM Serif Display:ital@0;1',
		'Open Sans:ital,wght@0,300;0,400;0,500;0,600;0,700;0,800;1,300;1,400;1,500;1,600;1,700;1,800'
	);

	$query_args = array(
		'family'	=> rawurlencode(implode('|',$font_family)),
	);
	$mizan_real_estate_font_url = add_query_arg($query_args,'//fonts.googleapis.com/css');
	return $mizan_real_estate_font_url;
	$contents = wptt_get_webfont_url( esc_url_raw( $$mizan_real_estate_fonts_url ) );
}

/**
 * Enqueue scripts and styles.
 */

function mizan_real_estate_scripts() {

	$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

	wp_enqueue_style( 'font-awesome', get_template_directory_uri() . '/third-party/font-awesome/css/font-awesome' . $min . '.css', '', '4.7.0' );

	$mizan_real_estate_fonts_url = mizan_real_estate_fonts_url();
	if ( ! empty( $mizan_real_estate_fonts_url ) ) {
		wp_enqueue_style( 'mizan-real-estate-google-fonts', $mizan_real_estate_fonts_url, array(), null );
	}

	wp_enqueue_style( 'dashicons' );

	wp_enqueue_style('bootstrap-style', get_template_directory_uri().'/css/bootstrap.css');

	// Theme stylesheet.
	wp_enqueue_style( 'mizan-real-estate-style', get_stylesheet_uri(), null, date( 'Ymd-Gis', filemtime( get_template_directory() . '/style.css' ) ) );

	wp_enqueue_style( 'mizan-real-estate-style', get_stylesheet_uri() );
	wp_style_add_data( 'mizan-real-estate-style', 'rtl', 'replace' );

	if (get_theme_mod('mizan_real_estate_animation', true) == true){
		wp_enqueue_script( 'mizan-real-estate-wow-js', get_template_directory_uri() . '/js/wow.js', array('jquery'),'' ,true );
		wp_enqueue_style( 'mizan-real-estate-animate-css', get_template_directory_uri().'/css/animate.css' );
	}

	require get_parent_theme_file_path( '/inc/color-palette/custom-color-control.php' );
	wp_add_inline_style( 'mizan-real-estate-style',$mizan_real_estate_color_palette_css );

	// Theme block stylesheet.
	wp_enqueue_style( 'mizan-real-estate-block-style', get_theme_file_uri( '/css/blocks.css' ), array( 'mizan-real-estate-style' ), '20211006' );
	
	wp_enqueue_script( 'mizan-real-estate-custom-js', get_template_directory_uri(). '/js/custom.js', array('jquery') ,'',true);
	
	wp_enqueue_script( 'jquery-superfish', get_theme_file_uri( '/js/jquery.superfish.js' ), array( 'jquery' ), '2.1.2', true );
	wp_enqueue_script('bootstrap-js', get_template_directory_uri().'/js/bootstrap.js', array('jquery'), '', true);
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
	wp_enqueue_style( 'mizan-real-estate-style', get_stylesheet_uri() );

	// body
	$mizan_real_estate_body_font_family = mizan_real_estate_get_option('mizan_real_estate_body_font_family');

	// H1
	$mizan_real_estate_h1_font_family = mizan_real_estate_get_option('mizan_real_estate_h1_font_family');
	$mizan_real_estate_h1_font_size   = mizan_real_estate_get_option('mizan_real_estate_h1_font_size');

	// H2
	$mizan_real_estate_h2_font_family = mizan_real_estate_get_option('mizan_real_estate_h2_font_family');
	$mizan_real_estate_h2_font_size   = mizan_real_estate_get_option('mizan_real_estate_h2_font_size');

	// H3
	$mizan_real_estate_h3_font_family = mizan_real_estate_get_option('mizan_real_estate_h3_font_family');
	$mizan_real_estate_h3_font_size   = mizan_real_estate_get_option('mizan_real_estate_h3_font_size');

	// H4
	$mizan_real_estate_h4_font_family = mizan_real_estate_get_option('mizan_real_estate_h4_font_family');
	$mizan_real_estate_h4_font_size   = mizan_real_estate_get_option('mizan_real_estate_h4_font_size');

	// H5
	$mizan_real_estate_h5_font_family = mizan_real_estate_get_option('mizan_real_estate_h5_font_family');
	$mizan_real_estate_h5_font_size   = mizan_real_estate_get_option('mizan_real_estate_h5_font_size');

	// H6
	$mizan_real_estate_h6_font_family = mizan_real_estate_get_option('mizan_real_estate_h6_font_family');
	$mizan_real_estate_h6_font_size   = mizan_real_estate_get_option('mizan_real_estate_h6_font_size');

	$mizan_real_estate_custom_css = '
		body{
		    font-family: '.esc_html($mizan_real_estate_body_font_family).';
		}

		h1{
		    font-family: '.esc_html($mizan_real_estate_h1_font_family).'!important;
		    font-size: '.esc_html($mizan_real_estate_h1_font_size).'px!important;
		}
		h2{
		    font-family: '.esc_html($mizan_real_estate_h2_font_family).'!important;
		    font-size: '.esc_html($mizan_real_estate_h2_font_size).'px!important;
		}
		h3{
		    font-family: '.esc_html($mizan_real_estate_h3_font_family).'!important;
		    font-size: '.esc_html($mizan_real_estate_h3_font_size).'px!important;
		}
		h4{
		    font-family: '.esc_html($mizan_real_estate_h4_font_family).'!important;
		    font-size: '.esc_html($mizan_real_estate_h4_font_size).'px!important;
		}
		h5{
		    font-family: '.esc_html($mizan_real_estate_h5_font_family).'!important;
		    font-size: '.esc_html($mizan_real_estate_h5_font_size).'px!important;
		}
		h6{
		    font-family: '.esc_html($mizan_real_estate_h6_font_family).'!important;
		    font-size: '.esc_html($mizan_real_estate_h6_font_size).'px!important;
		}
	';
	wp_add_inline_style('mizan-real-estate-style', $mizan_real_estate_custom_css);
}
add_action( 'wp_enqueue_scripts', 'mizan_real_estate_scripts' );

/*radio button sanitization*/
function mizan_real_estate_sanitize_choices( $input, $setting ) {
    global $wp_customize;
    $control = $wp_customize->get_control( $setting->id );
    if ( array_key_exists( $input, $control->choices ) ) {
        return $input;
    } else {
        return $setting->default;
    }
}

/**
 * Enqueue styles for the block-based editor.
 *
 * @since Mizan Real Estate
 */
function mizan_real_estate_block_editor_styles() {
	// Theme block stylesheet.
	wp_enqueue_style( 'mizan-real-estate-editor-style', get_template_directory_uri() . '/css/editor-blocks.css', array(), '20101208' );

	$mizan_real_estate_fonts_url = mizan_real_estate_fonts_url();
	if ( ! empty( $mizan_real_estate_fonts_url ) ) {
		wp_enqueue_style( 'mizan-real-estate-google-fonts', $mizan_real_estate_fonts_url, array(), null );
	}
}
add_action( 'enqueue_block_editor_assets', 'mizan_real_estate_block_editor_styles' );


function mizan_real_estate_theme_setup() {
/**
 * Load init.
 */
require_once get_template_directory() . '/inc/init.php';

// Dashboard Admin Info
require get_template_directory() . '/inc/dashboard-admin-info.php';

/**
 *  Webfonts 
 */
require_once get_template_directory() . '/inc/wptt-webfont-loader.php';

require_once get_template_directory() . '/inc/recommendations/tgm.php';

require_once get_template_directory() . '/inc/upsell/class-upgrade-pro.php';

require get_template_directory() . '/inc/getting-started/getting-started.php';

require get_template_directory() . '/inc/getting-started/plugin-activation.php';

define('MIZAN_REAL_ESTATE_DOCUMENTATION',__('https://preview.mizanthemes.com/setup-guide/mizan-real-estate-free/','mizan-real-estate'));
define('MIZAN_REAL_ESTATE_SUPPORT',__('https://wordpress.org/support/theme/mizan-real-estate/','mizan-real-estate'));
define('MIZAN_REAL_ESTATE_REVIEW',__('https://wordpress.org/support/theme/mizan-real-estate/reviews/','mizan-real-estate'));
define('MIZAN_REAL_ESTATE_BUY_NOW',__('https://www.mizanthemes.com/products/real-estate-wordpress-theme/','mizan-real-estate'));
define('MIZAN_REAL_ESTATE_LIVE_DEMO',__('https://preview.mizanthemes.com/mizan-real-estate/','mizan-real-estate'));
define('MIZAN_REAL_ESTATE_PRO_DOC',__('https://preview.mizanthemes.com/setup-guide/mizan-real-estate-pro/','mizan-real-estate'));

add_filter('mizan_importer_plugins_list', function ($plugins) {
		$mizan_real_estate_desired_order = ['woocommerce', 'yith-woocommerce-wishlist'];
		   foreach (['all', 'install', 'update', 'activate'] as $mizan_real_estate_section) {
			   if (!isset($plugins[$mizan_real_estate_section])) continue;
		$mizan_real_estate_reordered = [];
			   foreach ($mizan_real_estate_desired_order as $slug) {
				   if (isset($plugins[$mizan_real_estate_section][$slug])) {
		$mizan_real_estate_reordered[$slug] = $plugins[$mizan_real_estate_section][$slug];
		unset($plugins[$mizan_real_estate_section][$slug]);
		}
		}
		$plugins[$mizan_real_estate_section] = $mizan_real_estate_reordered + $plugins[$mizan_real_estate_section];
		}
		   return $plugins;
	});

}
add_action('after_setup_theme', 'mizan_real_estate_theme_setup');